/// Displaying events of calendar's in either today's events or monthly events ////
/// In an time ordered list (Horizontal-Vertical versions supported)

var list = {
    /// Default configuration of list-calendar ///
    default:{
        /// For testing ///
        player_params:{
            fonts:['./fonts/Montserrat.css']
        },
        widget:2,
        theme:'light',
        eventcolor:'#000000',
        activebackground:'#076890',
        eventbackground:'#EBEBEB',
        timeformat:24,
        dateformat:'DD/MM/YYYY',
        format:'day', // Format - "day" / "week" / "month"
        font:'Montserrat'
    },

    events:[],

    /// Initialize the list calendar ////
    init: function(config, events){
        config = this.parseColorFields(config);

        var last_day = this.getDay();
        events = this.applyRecurringEvents(events, last_day);

        events = this.filterEvents(events);
        events = this.sortEvents(events);

        this.events = events;

        if(this.events.length == 0){
            var noeventshtml = this.noEventsTemplate();
            $('#main').html(noeventshtml);
            $('body').css('background', '#d8d8d8');
            return;
        }

        this.render();
    },

    /// Render the calendar in DOM ///
    render: function(){
        reset();

        /// Create the container ////
        $('#main').html('<div class="list-events-container"></div>');

        var next;
        /// Render each event individually ///
        for(var i=0; i<this.events.length; i++){
            var current = false;
            var ev = this.events[i];

            ev['index-id'] = i; /// Assign a list-id to access items easier

            // Check if this is the current event item ///
            if(moment().isAfter(moment(ev.start)) && moment().isBefore(moment(ev.end))){
                current = true;
                ev['current-event'] = true;
            }

            /// Check if this is an all-day event - If hour:min start is same as end ///
            if(moment(ev.start).format('hh:mm') == moment(ev.end).format('hh:mm')){
                ev['all-day-event'] = true;
            }

            /// Add the event start and event end in time format ///
            ev['start_time'] = moment(ev.start).format('HH:mm');
            ev['end_time']   = moment(ev.end).format('HH:mm');

            /// Render event template ///
            $('.list-events-container').append(this.eventTemplate(ev));

            this.adjust(ev);
        }
        // this.adjust();

        this.scroll();


        /// Change the opacity of layers ///
        if($('.list-event[current-event="true"]').length>0){
            $('.list-event[current-event="true"]').prevAll('.list-event').addClass('past-event');
            $('.list-event[current-event="true"]').next('.list-event').addClass('upcoming-event');

            /// If there is an active event - hide the all-day events ///
            // $('.list-event[all-day-event="true"]').hide();
        }

        // Hide the "Next" label from all the events
        $('.list-event .next-event-label').hide();

        if($('.list-event[current-event="true"]').length == 0){
            this.showNext();
        }
    },

    // ---------------------------------
    // ----------- REFRESH FUNCTION ----------------
    // -- Removes the current event class from all elements
    // -- Adds the current event classees in the active events now
    // ---- NOTE : Called only from the start_widget timeCheck interval -----
    // ----------------------------------
    refresh: function(){
        $('.list-event').removeAttr('current-event').removeClass('past-event').removeClass('upcoming-event');

        for(var i=0; i<this.events.length; i++){
            var ev = this.events[i];
            // Check if this is the current event item ///
            if(moment().isAfter(moment(ev.start)) && moment().isBefore(moment(ev.end))){
                $('.list-event[event-id="' + ev['index-id'] + '"]').attr('current-event', true);
            }
        }

        /// Change the opacity of layers ///
        if($('.list-event[current-event="true"]').length>0){
            $('.list-event[current-event="true"]').prevAll('.list-event').addClass('past-event');
            $('.list-event[current-event="true"]').next('.list-event').addClass('upcoming-event');

            /// If there is an active event - hide the all-day events ///
            // $('.list-event[all-day-event="true"]').hide();

            this.scroll();
        }

        if($('.list-event[current-event="true"]').length == 0){
            this.showNext();
        }

        applyThemeStyle(window.config);
    },

    //// Show the Next event as active - with a right flag 'Next' ///
    showNext: function(){
        // Display the "Next" label at the upcoming event (if no active event)
        var next_hour,next_event_id;
        var now = moment().format('HH');
        var now_minutes = moment().format('mm');

        // Find the upcoming event //
        $('.list-event').each(function(){

            // Get the time of each event
            var time = $(this).attr('event-start');
            var minutes = parseInt(time.split(':')[1]);
            time = parseInt(time.split(':')[0]);

            /// If not today's event - ignore it ///
            if(!moment($(this).attr('event-date')).isSame(moment(), 'day')) return;

            /// Compare with time-now ///
            if(time < now) return;
            else if(time == now){
                if(minutes < now_minutes) return;
                else{
                    if(!next_hour || next_hour > time){
                        next_hour = time;
                        next_event_id = $(this).attr('event-id');
                    }
                }
            }
            else{
                if(!next_hour || next_hour > time){
                    next_hour = time;
                    next_event_id = $(this).attr('event-id');
                }
            }
        });

        /// apply styles - like it was the current event ///
        $('.list-event[event-id="' + next_event_id + '"]').attr('current-event', 'true').addClass('upcoming-event').find('.next-event-label').show();
        /// Apply styles in the Prev / Next  events ///
        $('.list-event[current-event="true"]').prevAll('.list-event').addClass('past-event');
        $('.list-event[current-event="true"]').next('.list-event').addClass('upcoming-event');
        /// Scroll to this event ///
        this.scroll();
    },

    scroll: function(){
        /// Scroll to the current event ///
        if($('.list-event[current-event="true"]').length > 0){
            var item_height = $('.list-event[current-event="true"]').height();
            var current_item_position = $('.list-event[current-event="true"]').position().top;


            var res = {width: $('body').width() , height: $('body').height()};
            var items = parseInt(res.height / item_height);

            $('html, body').animate({scrollTop: current_item_position - (item_height * items / 2) + item_height}, 1000, 'swing', function(){});
            // $(document).scrollTop($('.list-event[current-event="true"]').position().top - 90)
        }
    },

    // Use the 'rrule.js' library
    // To get the dates of the recurring event
    // Based on the Rules of the calendar event
    applyRecurringEvents: function(events,last_day){
        var rruleset = rrule.RRuleSet();

        var data = [];
        for(var i=0; i<events.length; i++){
            var event = events[i];
            var eventHour = moment(event.start).get('h');
            var eventMin  = moment(event.start).get('m');
            var eventEndHour = moment(event.end).get('h');
            var eventEndMin  = moment(event.end).get('m');

            // data.push(event); /// Push the original event ///

            if(event.recurrences){
                for(var re in event.recurrences){
                    var ev = event.recurrences[re];
                    ev['start'] = parseDate(ev.start);
                    ev['end']   = parseDate(ev.end);
                    ev['title'] = ev.summary;
                    ev['color'] = events[i].color;
                    ev['textColor'] = events[i].textcolor;
                    ev['location'] = ev.location;
                    ev['rrule'] = ev.rrule;
                    ev['recurrences'] = ev.recurrences;
                    data.push(ev);
                }
            }

            else if(event.rrule){
                var rule = event.rrule;
                if(rule.indexOf('UNTIL') < 0) rule += ';UNTIL=' + last_day.substring(0,8) + 'T060000';
                rule += ';DTSTART=' + moment(event.start).format('YYYYMMDD') + 'T060000';

                var dates = rruleset.RRule.fromString(rule).all();
                /// For every recursion of the event //
                for(var j=0; j<dates.length; j++){
                    /// Create a clone event ///
                    var temp_event = JSON.parse(JSON.stringify(event));

                    // Change it's start-date at the recursion-date //
                    temp_event.start = moment(dates[j]).set({h: eventHour, m:eventMin}).toString();

                    // Change it's end-date at the recursion-end-date //
                    temp_event.end = moment(dates[j]).set({h: eventEndHour, m:eventEndMin}).toString();

                    try{
                        /// Try to remove the GMT to place the time correctly ///
                        temp_event.start = temp_event.start.split('GMT')[0];
                    }
                    catch(err){}

                    try{
                        /// Try to remove the GMT from end-date to place the time correctly ///
                        temp_event.end = temp_event.end.split('GMT')[0];
                    }
                    catch(err){};

                    data.push(temp_event);

                }
            }
            else{
                /// non-recurring event - Just push to the list ///
                data.push(event);
            }
        }

        return data;
    },

    adjust:function(ev){

        var min = 4;
        var max_height = 192;
        var min_height = 120;

        var res = {width: $('body').width() , height: $('body').height()};
        var $ev = $('.list-event[event-id="' + ev['index-id'] + '"]');
        if(res.height < max_height * min){
            var new_height = res.height / 4;
            var pad_divide = max_height / new_height;
            var new_pad = 30/pad_divide;
            var new_time_pad = 12/pad_divide;
            // if(new_height < min_height) new_height = min_height;

            // $('.list-event-time').css('padding-top', new_time_pad + 'px');
            $ev.css('height', new_height).css('font-size', new_height + 'px').css('padding-top', new_pad + 'px');
        }



        var recursion = function(fsize){
            if(fsize < 50) return;


            $ev.css('font-size' , fsize + 'px');


            var nameheight = $ev.find('.list-event-name').height();
            // $('.list-event-name').each( function(){
            //     var nheight = $(this).height();
            //     console.log(nheight)
            //     if(nheight > nameheight) nameheight = nheight
            // })
            var paddingtop = new_pad || 30;
            var paddingbottom = 40;
            var cnameheight = $ev.find('.list-event-calendar-name').height();
            var timeheight  = $ev.find('.list-event-time').height();

            var container_height = $('.list-event').innerHeight();
            // debugger
            if(container_height < nameheight + paddingbottom + paddingtop + cnameheight + timeheight){
                recursion(--fsize)
            }else{

            }
        }

        var fontsize = new_height || 192;
        recursion(fontsize);
        // var
    },

    //// HTML template of every event ///
    eventTemplate: function(ev){
        var html = '';

        html += '<div class="list-event" event-id="' + ev['index-id'] + '" event-date="' + ev['start'] + '" event-start="' + ev['start_time'] + '" event-end="' + ev['end_time'] + '"';
        /// All-day events - will never be considered as active ///
        if(ev['current-event'] && !ev['all-day-event']) html += ' current-event=true';
        if(ev['all-day-event']) html += ' all-day-event=true';
        html += '>';

        try{
            /// Try to uppercase the calendar name ///
            ev.calendar = ev.calendar.toUpperCase();
        }
        catch(err){};

        html +=  '<div class="list-event-calendar-name">' + ev.calendar + '</div>';
        html +=  '<div class="list-event-name">' + ev.title + '</div>';

        var start = (window.config.timeformat == 12) ? moment(ev.start).format('hh:mm A') : moment(ev.start).format('HH:mm');
        var end   = (window.config.timeformat == 12) ? moment(ev.end).format('hh:mm A') : moment(ev.end).format('HH:mm');

        html +=  '<div class="list-event-time">';

        // Include the date if the format is monthly //
        if(window.config.format == 'month' || window.config.format == 'week'){
            html += moment(ev.start).format(window.config.dateformat) + ' ';
        }
        html += start + ' - ' + end + '</div>';

        /// Upcoming event text -- Will be displayed only in the next event ///
        html += '<span class="next-event-label"> Next </span>';

        html += '</div>';

        return html;
    },

    //// HTML template when there are no scheduled events to display ///
    noEventsTemplate: function(){
        var html = '\
            <div class="list-no-events-container">\
                <div class="list-no-events">\
                    <div class="list-no-events-text"> No events scheduled </div>\
                </div>\
            </div>\
        ';

        return html;
    },

    /// Filter the events to get only the current day's events ////
    ///// -- Check format (monthly/today) in the configuration -- /////
    filterEvents: function(events){
        var out = [];

        for(var i=0; i<events.length; i++){
            var eventDate = events[i].start;
            if(window.config.format == 'month'){
                // Monthly events //
                if(moment(eventDate).isSame(moment(), 'month')){
                    out.push(events[i]);
                }
            }
            else if(window.config.format == 'week'){
                /// Weekly Events ///
                if(moment(eventDate).isSame(moment(), 'week')){
                    out.push(events[i]);
                }
            }
            else{
                // Today's events //
                if(moment(eventDate).isSame(moment(), 'day')){
                    out.push(events[i]);
                }
            }
        }

        /// Remove duplicated events ////
        var filtered = [];
        for(var i=0; i<out.length; i++){
            var exists = false;
            for(var j=0; j<filtered.length; j++){
                if(filtered[j].title == out[i].title && ((filtered[j].start && filtered[j].start == out[i].start) || (filtered[j].end && filtered[j].end == out[i].end))){
                    exists = true;
                }
            }

            if(!exists) filtered.push(out[i]);
        }

        return filtered;
    },

    /// Sort the events based on the start-time //
    sortEvents: function(events){
        return events.sort(function(e1, e2){
            return moment(e1.start).isBefore(e2.start) ? -1 : 1;
        })
    },

    /// Get day in moment-timezone format ///
    getDay: function(){
        if(window.config.format == 'month'){
            var date = moment().clone().endOf('month').format().split('-').join('').split(':').join('');
        }
        else if(window.config.format == 'week'){
            var date= moment().clone().weekday(6).format().split('-').join('').split(':').join('')
        }
        else{
            var date = moment().format().split('-').join('').split(':').join('');
        }
        date = date.substr(0, date.length - 5);
        return date;
    },

    //// Checking for color fields to parse them in RGBA format ////
    parseColorFields: function(config){
        try{ config.activebackground = lib.hextoRGBA(config.activebackground)}
        catch(err){};

        try{ config.eventcolor = lib.hextoRGBA(config.eventcolor)}
        catch(err){};

        try{ config.eventbackground = lib.hextoRGBA(config.eventbackground)}
        catch(err){};

        window.config = config;
        return config;
    },

    //// Create a default_theme clone object for the List widget
    //// Based on the values from configuration
    getCustomTheme: function(){
        config = window.config;

        /// Clone the default_theme object ///
        var custom = JSON.parse(JSON.stringify(default_theme));

		custom['--list-events-active-background'] = config.activebackground;
		custom['--list-events-active-color'] = config.eventcolor;
		custom['--list-events-background'] = config.eventbackground;
        custom['--list-events-textcolor'] = config.eventcolor;

        return custom;
    }

}