/// This is the agenda calendar widget
/// With 2-options provided (weekly / daily)

var agenda = {

    //// Default Agenda configuration ///
    default:{/// For testing ///
        player_params:{
            fonts:['./fonts/Montserrat.css']
        },
        agendaType:'week',
        theme:'light',
        font:'Montserrat',
        widget:0
    },
    calendar_config : {},
    events: [],

    init: function(config, events){
        config = this.parseColorFields(config);

        /// Get the recurring events ///
        var last_day = this.getLastDay(); /// Get the last day of the calendar
        events = this.applyRecurringEvents(events, last_day);

        events = this.filterEvents(config,events);
        events = this.sortEvents(events);

        this.events = events;

        this.render();
    },

    render: function(){
        reset();

        this.calendar_config = this.getConfig(events);
        this.calendar_config.events = this.events;

        /// Create the calendar in DOM ///
        $('#main').fullCalendar(this.calendar_config);

        /// Remove the toolbar ///
        $('.fc-header-toolbar').remove();

        if($('body').width() < 600){
            $('.fc-agenda-view .fc-day-header').each(function(){
                var text = $(this).find('span').text();
                $(this).find('span').text(text.split(' ')[0]);
            })
        }
    },

    // Use the 'rrule.js' library
    // To get the dates of the recurring event
    // Based on the Rules of the calendar event
    applyRecurringEvents: function(events,last_day){
        var rruleset = rrule.RRuleSet();

        var data = [];
        for(var i=0; i<events.length; i++){
            var event = events[i];
            var eventHour = moment(event.start).get('h');
            var eventMin  = moment(event.start).get('m');
            var eventEndHour = moment(event.end).get('h');
            var eventEndMin  = moment(event.end).get('m');

            if(eventHour == eventEndHour && eventMin == eventEndMin) continue;
            // data.push(event); /// Push the original event ///

            if(event.recurrences){
                for(var re in event.recurrences){
                    var ev = event.recurrences[re];
                    ev['start'] = parseDate(ev.start);
                    ev['end']   = parseDate(ev.end);
                    ev['title'] = ev.summary;
                    ev['color'] = events[i].color;
                    ev['textColor'] = events[i].textcolor;
                    ev['location'] = ev.location;
                    ev['rrule'] = ev.rrule;
                    ev['recurrences'] = ev.recurrences;
                    data.push(ev);
                }
            }

            else if(event.rrule){
                var rule = event.rrule;
                if(rule.indexOf('UNTIL') < 0) rule += ';UNTIL=' + last_day.substring(0,8) + 'T060000';
                rule += ';DTSTART=' + moment(event.start).format('YYYYMMDD') + 'T060000';

                var dates = rruleset.RRule.fromString(rule).all();
                /// For every recursion of the event //
                for(var j=0; j<dates.length; j++){
                    /// Create a clone event ///
                    var temp_event = JSON.parse(JSON.stringify(event));

                    // Change it's start-date at the recursion-start-date //
                    temp_event.start = moment(dates[j]).set({h: eventHour, m:eventMin}).toString();

                    // Change it's end-date at the recursion-end-date ///
                    temp_event.end = moment(dates[j]).set({h: eventEndHour, m:eventEndMin}).toString();

                    try{
                        /// Try to remove the GMT to place the time correctly ///
                        temp_event.start = temp_event.start.split('GMT')[0];
                    }
                    catch(err){}

                    try{
                        /// Try to remove the GMT from end-date to place the time correctly ///
                        temp_event.end = temp_event.end.split('GMT')[0];
                    }
                    catch(err){};


                    data.push(temp_event);

                }
            }
            else{
                /// non-recurring event - Just push to the list ///
                data.push(event);
            }
        }

        return data;
    },

    /// Filters the events that will be used in this widget
    filterEvents: function(config,events){
        var out = [];
        if(config.agendaType == 'week'){
            // Weekly Agenda //
            for(var i=0; i<events.length; i++){
                var eventDate = events[i].start;
                if(moment(eventDate).isSame(moment(), 'week')){
                    /// Checks if this event belongs to this week ////
                    out.push(events[i]);
                }
            }
        }
        else{
            // Daily Agenda //
            for(var i=0; i<events.length; i++){
                var eventDate = events[i].start;
                console.log(eventDate);
                if(moment(eventDate).isSame(moment(), 'day')){
                    /// Checks if this event belongs to this day ///
                    out.push(events[i]);
                }
            }
        }

        /// Remove duplicated events ////
        var filtered = [];
        for(var i=0; i<out.length; i++){
            var exists = false;
            for(var j=0; j<filtered.length; j++){
                if(filtered[j].title == out[i].title && ((filtered[j].start && filtered[j].start == out[i].start) || (filtered[j].end && filtered[j].end == out[i].end))){
                    exists = true;
                }
            }

            if(!exists) filtered.push(out[i]);
        }
        return filtered;
    },

    /// Get the filtered events that will be used - and sort them by start Date ///
    sortEvents: function(events){
        return events.sort(function(e1, e2){
            return moment(e1.start).isBefore(e2.start) ? -1 : 1;
        })
    },

    /// Displaying the best timeslot duration based on the window resolution and the time start/end of the events ///
    getConfig: function(events, increaseTimeslot){
        /// Default timeslot duration at 1-hour ///
        var timeSlotDuration = '00:60:00';
        if(increaseTimeslot) timeSlotDuration = '02:00:00';

        // These are the pixels of every row height ///
        var default_row_height = 45;
        if(increaseTimeslot) default_row_height = 22.5; /// Do the maths with half sized rows - cause we increase the timeslotDuration at 2-hours ////

        var res = { width: $('body').width(), height: $('body').height()};

        /// How many rows can fit in the screen ///
        var rows_to_fit = Math.round(res.height / default_row_height) - 1; /// Minus one for the header which has the same height ////

        var min = this.getLowestHour(events);
        var max = this.getHighestHour(events);

        var rows_to_display = max-min;

        if(rows_to_display <= rows_to_fit){
            /// We have more space ///
            var extra_rows = rows_to_fit - rows_to_display;
            for(var i=0; i<extra_rows; i++){
                /// Start decreasing the min-hour - and increasing the max-hour - One at a time ///
                if(i % 2 == 0){
                    if(min > 6) min --;
                    else if(max < 29) max ++; /// max 29 to check until 5:00 AM
                }
                else{
                    if(max < 29) max ++; /// max 29 to check until 5:00 AM
                    else if(min > 6) min --;
                }
            }
        }
        else{
            if(!increaseTimeslot) return this.getConfig(events, true);
            /// There is no space to display all the hours ////
            timeSlotDuration = '02:00:00';
        }

        /// Setting the view of the agenda ///
        var defaultView = config.agendaType == 'week' ? 'agendaWeek' : 'agendaDay';

        return{
            plugin: 'agenda',
            defaultView: defaultView,
            slotDuration: timeSlotDuration,
            minTime: intToTime(min),
            maxTime: intToTime(max),
            agendaEventMinHeight:20,
            slotLabelFormat:'hh:mm A',
            allDaySlot: false
        }

    },

    /// Get's the lowest event-start hour - Starting from 06:00 AM ///
    getLowestHour: function(events){
        var hour =  24;

        for(var i=0; i<events.length; i++){
            var eventStartHour = Math.round(moment(events[i].start).format('HH'));
            if(eventStartHour < hour) hour = eventStartHour;
        }

        return hour;
    },

    /// Get's the hightest event-start OR event-end hour ////
    getHighestHour: function(events){
        var hour = 0;

        for(var i=0; i<events.length; i++){
            var eventStartHour = Math.round(moment(events[i].start).format('HH'));
            var eventEndHour   = Math.round(moment(events[i].end).format('HH'));
            if(eventStartHour > hour) hour = eventStartHour;
            if(eventEndHour   > hour) hour = eventEndHour;
        }

        if(hour < 24) hour ++; /// add one more to make sure it will cover the event-end at half-hour ///

        return hour;
    },

    /// Get the last day of the calendar ///
    getLastDay: function(){
        /// Return the next 'Saturday' - that's the last day column of the agenda week widget ///
        var date= moment().clone().weekday(6).format().split('-').join('').split(':').join('')
        date = date.substr(0, date.length -5);
        return date;
    },

    /// Return the default theme with color configuration fields applied ////
    getCustomTheme: function(){
        /// Agenda has no 'custom' theme property yet
        /// If we add color fields and custom config fields
        /// This function will handle it
        return default_theme;
    },

    parseColorFields: function(config){
        try{ config.textcolor = lib.hextoRGBA(config.textcolor)}
        catch(err){};

        window.config = config;

        return config;
    }

}