/**
 * ------ getImage function --------
 * @param {*url string} Gets the url of the article
 *
 * @param {*proxy server_address} localhost/get_article_image
 * ============== OUTPUT ==============
 *  @param {*image string} Returns the url of the original article image
 *  @param {*error If there was an error on the request proxy made to get image}
 *  @param {*proxy_error If there was an error inside the proxy request (Uncaught errors OR Syntax errors )}
 */
function getImage(articleUrl,callb){
    // ============================ PROXY ENDPOINT URL ============================= //
    // var url = 'http://10.20.110.163:9999/get_article_image';
    var url = 'http://localhost:9999/get_article_image';

    $.ajax({
        url:url,
        method:'POST',
        contentType: 'application/json',
        data:JSON.stringify({url:articleUrl}),
        timeout:10000,
        success:function(response){
            if('error' in response){
                callb(response.error);
                return;
            }
            else if('proxy_error' in response){
                callb(response.proxy_error);
                return;
            }
            else{
                if('image' in response){
                    callb(null,response.image);
                    return;
                }
                else{
                    callb('No image found')
                }
            }
        },
        error:function(err){
            console.error(err);
            callb(err);
        }
    })
}

/**
 *  /=========================== Request XML data ==============================/
 * ---------------------- Return the XML page -------------------
 */
function requestXMLdata(callb,refresh){
    var url = window.feed_url

    $.ajax({
        url:url,
        success:function(data){
            try{
                data = new XMLSerializer().serializeToString(data);
            }
            catch(err){
                console.log('Error on serialize XML to String');
                if(!refresh) error('url_no_xml'); /// display error page only on initialize
                callb('Url no XML')
            }
            callb(null, data);
        },
        error: function(err){
            console.log('Error on request XML');
            console.log(err);
            if(!refresh){
                /// display error page only on initialize
                if(err && err == 404) error('invalid_url');
                else error('connection_error');
            }
            callb(err);
        }
    })
}

/**
 * /======================== Validate XML through parser.js library ========================/
 */
function validateXML(xml){
    return parser.validate(xml);
}

/**
 * /======================== Check for "New" posts in the XML feed ========================/
 * @param{*items} Get the items from the requestXML
 * @output{*Boolean} :
 *      'false' if there are no new items in the XML
 *      'true' if there is an update on the News feed
 */
function xmlUpdated(items){
    var links = window.articleLinks;
    for(var i=0; i<items.length; i++){
        if(i >= window.options.number_of_slides) continue;
        var isNew = true;
        for(var j=0; j<links.length; j++){
            if(links[j] == items[i]){
                isNew = false;
                j = links.length; // go to next one
            }
        }
        if(isNew) return true; // update the feed
    }

    return false; /// no updates on the XML
}


/**
 * /============================ Error handling function  =======================/
 *  ---------- Getting parameter error code and displays the right error message --------------
 */
function error(code){
    var title = '' , errmsg = '';
    switch(code){
        case 'invalid_url':
            /// invalid RSS feed URL provided ////
            title  = 'Unable to load feed';
            errmsg = 'There was an error on loading feed data.Make sure the RSS feed URL you provided is correct and retry.';
            break;
        case 'url_no_xml':
            /// URL provided is not RSS ///
            title = 'Unable to load feed';
            errmsg = 'URL provided does not redirect to an RSS news feed.';
            break;
        case 'connection_error':
            /// Failed to connect in proxy/url given or any other connection error inside the widget execution ////
            title = 'Connection error';
            errmsg = 'There was a connectivity error during the widget loading process. Make sure your connection is right and retry in a few minutes.';
            break;
    }

    $('.error #error-page .error-header').html(title);
    $('.error #error-page .error-text-message').html(errmsg);

    $('.mask').hide()
    $('.error').show()
}

/**
 *  Horizontal alignment in feed  *
 * @param {*Feed Object} feed
 * @param {*Index for id attribute } idx
 */
function horizontal(feed, idx){
    if(feed.description !== '') feed.description = '“' + feed.description + '”';

    var html =  '';
    html += '<div class="feed" id="feed-' + idx + '">';

    if('image' in feed && feed.image !== ''){
        html += '<div class="image-container">';
        if(window.logo && window.logo !== ''){
            html += '<img class="logo-thumbnail" src="' + window.logo +'" />';
        }
        html +=  '<img style="width: 100%; height: auto;" src="' + feed.image + '" />';
        html += '</div>';

        html += '<div class="feed-container">';
        // if('feedImage' in feed){
        //     html += '<div class="feed-image-container">';
        //     html +=  '<img width="100" height="75" src="' + feed.feedImage + '" />';
        //     html += '</div>';
        // }
        html +=  '<h3 class="feed-title">' + feed.title + '</h3>';

        // if('media:thumbnail' in feed){
        //     html += '<div class="feed-thumbnail-container">'
        //     html +=  '<img width="' + feed['media:thumbnail']['@_width'] + '" height="' + feed['media:thumbnail']['@_height'] + '" src="' + feed['media:thumbnail']['@_url'] + '" />';
        //     html += '</div>';
        // }

        if('pubDate' in feed && feed.pubDate !== ''){
            var ago = postHours(feed.pubDate);
            html += '<span class="feed-date"> <i class="fa fa-clock"> </i> ' + ago.val + ' ' + ago.label + '</span>';
        }

        html +=  '<div class="feed-description-container">';
        html +=   '<span class="feed-description">' + feed.description + ' </span>';
        html +=  '</div>';


        html += '</div>';

    }
    else{
        html +=  '<h3 class="feed-title">' + feed.title + '</h3>';
        if('pubDate' in feed && feed.pubDate !== ''){
            var ago = postHours(feed.pubDate);
            html += '<span class="feed-date"> <i class="fa fa-clock"> </i>' + ago.val + ' ' + ago.label +  '</span>';
        }
        html +=  '<div class="feed-description-container">';
        html +=   '<span class="feed-description">' + feed.description + '</span>';
        if('image' in feed && feed.image !== ''){
            html += '<img src="' + feed.image + '" / style="max-width: 100%;">';
        }
        html +=  '</div>';
    }

    html += '</div>';

    return html;
}

/**
 * Vertical Alignment in feed
 * @param {*Feed object} feed
 * @param {*Index for id attribute } idx
 */
function vertical(feed,idx){
    if(feed.description !== '') feed.description = '“' + feed.description + '”';

    var html = '';

    html += '<div class="feed vertical" id="feed-' + idx + '">';

    if('image' in feed && feed.image !== ''){
        html += '<div class="col-lg-6 feed-container">';
        if('feedImage' in feed){
            html += '<div class="feed-image-container">';
            html +=  '<img src="' + feed.feedImage + '" />';
            html += '</div>';
        }
        html +=  '<h3 class="feed-title">' + feed.title + '</h3>';

        // if('media:thumbnail' in feed){
        //     html += '<div class="feed-thumbnail-container">'
        //     html +=  '<img width="' + feed['media:thumbnail']['@_width'] + '" height="' + feed['media:thumbnail']['@_height'] + '" src="' + feed['media:thumbnail']['@_url'] + '" />';
        //     html += '</div>';
        // }

        html +=  '<div class="feed-description-container">';
        html +=   '<span class="feed-description">' + feed.description + '</span>';
        html +=  '</div>';

        if('pubDate' in feed && feed.pubDate !== ''){
            var ago = postHours(feed.pubDate);
            html += '<span class="feed-date"> <i class="fa fa-clock"> </i> '+ ago.val + ' ' + ago.label + '</span>';
        }

        html += '</div>';

        html += '<div class="col-lg-6 image-container">';
        html +=  '<img style="width: 100%;" src="' + feed.image + '" />';
        html += '</div>';

    }
    else{
        html +=  '<h3 class="feed-title">' + feed.title + '</h3>';
        if('pubDate' in feed && feed.pubDate !== ''){
            var ago = postHours(feed.pubDate);
            html += '<span class="feed-date"> <i class="fa fa-clock"></i> ' + ago.val + ' ' + ago.label + '</span>';
        }
        html +=  '<div class="feed-description-container">';
        html +=   '<span class="feed-description">' + feed.description + '</span>';
        if('image' in feed && feed.image !== ''){
            html += '<img src="' + feed.image + '" / style="max-width: 100%;">';
        }
        html +=  '</div>';
    }

    html += '</div>';

    return html;
}


/**
 * Append styles from configuration
 */
function applyConfigurationStyles(container){
    var config = window.options;
    if('font_size' in config){
        $('.feed-title').css('font-size', config.font_size + 'px');
        $('.feed-description-container').css('font-size', (config.font_size - 8) + 'px');
        $('.feed-date').css('font-size', (config.font_size - 15) + 'px');
    }

    if('font_color' in config){
        $('.feed').css('color', HexToRGBA(config.font_color));
    }
    if('rgba_color' in config){
        $('body, .feed').css('background' , HexToRGBA(config.rgba_color));
    }

    //// maximum description length ////
    var desc = $(container + ' .feed-description').text();
    if(desc.length > 400){
        desc = desc.substr(0,400) + '...';
        $(container + ' .feed-description').text(desc);
    }

    //// maximum title length ////
    var title = $(container + ' .feed-title').text();
    if(title.length > 120){
        title = title.substr(0, 120)  + '...';
        $( container + ' .feed-title').text(title);
    }
}

//// get how many hours was the post uploaded /////
function postHours(date){
    var now = moment();
    date = moment(date);

    var diff = moment(now.diff(date)).format('hh:mm:ss');
    var arr = diff.split(':');

    var h = parseInt(arr[0]);
    if(h > 0) return { label:'Hours ago', val:h}

    var m = parseInt(arr[1]);
    if(m > 0) return { label:'Minutes ago' , val: m}

    var s = parseInt(arr[2]);
    if(s > 0) return { label:'Seconds ago', val:s}

    return 0
}

//// removes encoded HTML characters ////
function strip(html){
    var doc = new DOMParser().parseFromString(html, 'text/html');
    return doc.body.textContent || "";
}

//// removes all HTML tag elements from text /////
function stripHTML(my_string){
    var charArr   = my_string.split(''),
        resultArr = [],
        htmlZone  = 0,
        quoteZone = 0;
    for( x=0; x < charArr.length; x++ ){
     switch( charArr[x] + htmlZone + quoteZone ){
       case "<00" : htmlZone  = 1;break;
       case ">10" : htmlZone  = 0;resultArr.push(' ');break;
       case '"10' : quoteZone = 1;break;
       case "'10" : quoteZone = 2;break;
       case '"11' :
       case "'12" : quoteZone = 0;break;
       default    : if(!htmlZone){ resultArr.push(charArr[x]); }
     }
    }
    return resultArr.join('');
}

function HexToRGBA(hex){
    var r = parseInt(hex.substr(0,2),16);
    var g = parseInt(hex.substr(2,2),16);
    var b = parseInt(hex.substr(4,2),16);
    var a = Math.round(parseInt(hex.substr(6,2),16) / 2.55)/100;

    if(isNaN(r+g+b+a)) return hex;

    var rgba='rgba('+r+','+g+','+b+','+a+')'

    return rgba;
}
